<?php
/**
 * @copyright 2010-2013 JTL-Software GmbH
 * @package jtl\Connector\Application
 */

namespace jtl\Connector\Application\Error;

use jtl\Connector\Core\Http\Response;
use jtl\Connector\Core\Logger\Logger;
use jtl\Connector\Core\Rpc\Error;
use jtl\Connector\Core\Rpc\RequestPacket;
use jtl\Connector\Core\Rpc\ResponsePacket;
use jtl\Connector\Core\Utilities\RpcMethod;
use jtl\Connector\Event\EventHandler;
use Symfony\Component\EventDispatcher\EventDispatcher;
use jtl\Connector\Formatter\ExceptionFormatter;

class ErrorHandler implements IErrorHandler
{
    /**
     * @var \Symfony\Component\EventDispatcher\EventDispatcher
     */
    protected $eventDispatcher;

    public function __construct()
    {
        // Exception
        $this->setExceptionHandler($this->getExceptionHandler());

        // Error
        $this->setErrorHandler($this->getErrorHandler());

        // Shutdown
        $this->setShutdownHandler($this->getShutdownHandler());
    }

    protected function triggerRpcAfterEvent($data, $method)
    {
        if ($this->eventDispatcher !== null) {
            $method = RpcMethod::splitMethod($method);
            EventHandler::dispatchRpc($data, $this->eventDispatcher, $method->getController(), $method->getAction(), EventHandler::AFTER);
        }
    }

    public function setExceptionHandler(callable $func)
    {
        set_exception_handler($func);
    }

    public function getExceptionHandler()
    {
        return function ($e) {
            $error = new Error();
            if (self::isThrowable($e)) {
                $trace = $e->getTrace();
                if (isset($trace[0]['args'][0])) {
                    $requestpacket = $trace[0]['args'][0];
                }
    
                $error->setCode($e->getCode())
                    ->setData(ExceptionFormatter::format($e))
                    ->setMessage($e->getMessage());
    
                Logger::write($error->getData(), Logger::ERROR, Logger::CHANNEL_GLOBAL);
            }

            $responsepacket = new ResponsePacket();
            $responsepacket->setError($error)
                ->setId('unknown')
                ->setJtlrpc('2.0');

            $method = 'unknown.unknown';
            if (isset($requestpacket) && $requestpacket !== null && is_object($requestpacket) && $requestpacket instanceof RequestPacket) {
                $responsepacket->setId($requestpacket->getId());
                $method = $requestpacket->getMethod();
            }

            $this->triggerRpcAfterEvent($responsepacket->getPublic(), $method);
            Response::send($responsepacket);
        };
    }

    public function setErrorHandler(callable $func)
    {
        set_error_handler($func);
    }

    public function getErrorHandler()
    {
        return function ($errno, $errstr, $errfile, $errline) {
            $types = [
                E_ERROR => [Logger::ERROR, 'E_ERROR'],
                E_WARNING => [Logger::WARNING, 'E_WARNING'],
                E_PARSE => [Logger::WARNING, 'E_PARSE'],
                E_NOTICE => [Logger::NOTICE, 'E_NOTICE'],
                E_CORE_ERROR => [Logger::ERROR, 'E_CORE_ERROR'],
                E_CORE_WARNING => [Logger::WARNING, 'E_CORE_WARNING'],
                E_COMPILE_ERROR => [Logger::ERROR, 'E_COMPILE_ERROR'],
                E_USER_ERROR => [Logger::ERROR, 'E_USER_ERROR'],
                E_USER_WARNING => [Logger::WARNING, 'E_USER_WARNING'],
                E_USER_NOTICE => [Logger::NOTICE, 'E_USER_NOTICE'],
                E_STRICT => [Logger::NOTICE, 'E_STRICT'],
                E_RECOVERABLE_ERROR => [Logger::ERROR, 'E_RECOVERABLE_ERROR'],
                E_DEPRECATED => [Logger::INFO, 'E_DEPRECATED'],
                E_USER_DEPRECATED => [Logger::INFO, 'E_USER_DEPRECATED']
            ];

            if (isset($types[$errno])) {
                $err = "(" . $types[$errno][1] . ") File ({$errfile}, {$errline}): {$errstr}";
                Logger::write($err, $types[$errno][0], Logger::CHANNEL_GLOBAL);
            } else {
                Logger::write("File ({$errfile}, {$errline}): {$errstr}", Logger::ERROR, Logger::CHANNEL_GLOBAL);
            }
        };
    }

    public function setShutdownHandler(callable $func)
    {
        register_shutdown_function($func);
    }

    public function getShutdownHandler()
    {
        return function () {
            if (($err = error_get_last())) {
                $allowed = [
                    E_ERROR,
                    E_CORE_ERROR,
                    E_USER_ERROR,
                    E_RECOVERABLE_ERROR,
                    E_COMPILE_ERROR,
                    E_PARSE
                ];

                if (in_array($err['type'], $allowed)) {
                    ob_clean();

                    $error = new Error();
                    $error->setCode($err['type'])
                        ->setData('Shutdown! File: ' . $err['file'] . ' - Line: ' . $err['line'])
                        ->setMessage($err['message']);

                    Logger::write(sprintf(
                        '%s - Type: %s - Message: %s',
                        $error->getData(),
                        $err['type'],
                        $error->getMessage()
                    ), Logger::ERROR, 'global');

                    $responsepacket = new ResponsePacket();
                    $responsepacket->setError($error)
                        ->setId('unknown')
                        ->setJtlrpc('2.0');

                    $this->triggerRpcAfterEvent($responsepacket->getPublic(), 'unknown.unknown');

                    Response::send($responsepacket);
                }
            }
        };
    }
    
    /**
     * @param EventDispatcher $dispatcher
     * @return ErrorHandler
     */
    public function setEventDispatcher(EventDispatcher $dispatcher)
    {
        $this->eventDispatcher = $dispatcher;
        return $this;
    }
    
    /**
     * @param mixed $e
     * @return bool
     */
    public static function isThrowable($e)
    {
        return version_compare(phpversion(), '7.0.0', '<') ?
            ($e instanceof \Exception) :
            ($e instanceof \Throwable);
    }
}
